/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: CifxConsole_nxDrvAPI.cpp 13941 2021-02-08 09:28:41Z RMayer $:

  Description:
    Test nxDrv API functions

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2018-10-19  Ported from cifXTest_Console V1.0.6.0
    2018-09-11  initial version

**************************************************************************************/
#include "OS_Includes.h"
#include "CifXConsole.h"
#include "netXAPI.h"

/* browse devices (nxDrvBrowseDevices()) callback wrapper structure */
typedef struct BROWSE_DEV_CB_Ttag
{
  PFN_NXAPI_BROWSE_CALLBACK pfnCallback;  /*!< User callback  */
  void*                     pvUser;       /*!< User parameter */

} BROWSE_DEV_CB_T;

extern BROWSE_DEV_CB_T tMyAPP_ClbkInfo = {0};

/* Browse devices callback */
void APIENTRY MyBrowseCallback (uint32_t  ulBoard, BOARD_INFORMATION* ptBoardInfo, uint32_t ulStep, uint32_t ulMaxStep, void* pvUser, char bFinished, int32_t lError)
{
  UNREFERENCED_PARAMETER( ulStep);
  UNREFERENCED_PARAMETER( ulMaxStep);
  UNREFERENCED_PARAMETER( pvUser);
  UNREFERENCED_PARAMETER( lError);

  /* Display information */
  if( !bFinished)
  {
    printf("MyBrowseCallback() information\r\n");
    printf("          Board Name : %s\r\n", ptBoardInfo->abBoardName);
    printf("          Board Alias: %s\r\n", ptBoardInfo->abBoardAlias);
    printf("          ulBoard    : %d\r\n", ulBoard);
  }
}

/* Progress callback */
void APIENTRY MyProgressCallback(uint32_t ulStep, uint32_t ulMaxStep, void* pvUser, char bFinished, int32_t lError)
{
  UNREFERENCED_PARAMETER( pvUser);

  /* Display information */
  printf("MyProgressCallback() information\r\n");
  printf("          ulMaxStep : %d\r\n",      ulMaxStep);
  printf("          ulStep    : %d\r\n",      ulStep);
  printf("          bFinished : %d\r\n",      bFinished);
  printf("          lError    : 0x%.8x\r\n",  lError);
}



/*****************************************************************************/
/*! Test NXDrv API functions
 *  \param  hDevice  Device handle (unused)
 *  \return True on success                                                  */
/*****************************************************************************/
bool nxDrvAPITest( char* pszDeviceName)
{
  int32_t lRet   = NXAPI_NO_ERROR;
  bool    fFound = false;

  printf( "\r\n");
  printf( "-- nxDrvAPI Test \r\n");

  printf( "nxDrvInit()\r\n");

  /*----------------------*/
  /* Initialize nxDrv API */
  /*----------------------*/
  if ( NXAPI_NO_ERROR != (lRet = nxDrvInit()))
  {
    ShowError(lRet);
  }else
  {
    uint32_t                  ulCmd       = NXDRV_FIND_FIRST;
    uint32_t                  ulSearchIdx = 0;
    NXDRV_HW_INFORMATION      tDrvInfo;
    NXDRV_DEVICE_INFORMATION  m_tDeviceInfo;

    memset( &tDrvInfo,      0, sizeof( NXDRV_HW_INFORMATION));
    memset( &m_tDeviceInfo, 0, sizeof( NXDRV_DEVICE_INFORMATION));

    printf( "nxDrv API initialized\r\n");

    /*-------------------------*/
    /* Read driver information */
    /*-------------------------*/
    if( NXAPI_NO_ERROR != (lRet = nxDrvGetInformation( sizeof( tDrvInfo),
                                                       &tDrvInfo)))
    {
      ShowError( lRet);
    }else
    {
      printf( "\r\n");
      printf( "nxDrvGetInformation()\r\n");
      printf( "  Driver Name   : %s\r\n",     tDrvInfo.szDriverName);
      printf( "  Driver Version: %s\r\n",     tDrvInfo.szVersion);
      printf( "  Driver Type   : 0x%.3X\r\n", tDrvInfo.ulDriverType);
      printf( "  Device Class  : 0x%.3X\r\n", tDrvInfo.ulDeviceClass);
    }

    /*------------------------------*/
    /* Browse for available devices */
    /*------------------------------*/
    printf( "\r\n");
    printf( "nxDrvBrowseDevices()\r\n");
    if( NXAPI_NO_ERROR != (lRet = nxDrvBrowseDevices( MyBrowseCallback, &tMyAPP_ClbkInfo)))
    {
      ShowError( lRet);
    }

    /*------------------------------*/
    /* Execute an hardware reset    */
    /* On the given device          */
    /*------------------------------*/
    /* Search the given device in m_tDeviceInfo */
    printf( "\r\n");
    printf( "nxDrvFindDevice(), device with name: %s\r\n", pszDeviceName);
    printf( "  Command: NXDRV_FIND_FIRST = %d\r\n", ulCmd);
    do
    {
      printf( "  Search index = %d\r\n", ulSearchIdx);
      if( NXAPI_NO_ERROR != (lRet = nxDrvFindDevice(ulCmd, sizeof(m_tDeviceInfo), &m_tDeviceInfo, &ulSearchIdx)))
      {
        ShowError(lRet);
      } else
      {
        // Check if we found our device
        long lStingLenMax = max( strlen(m_tDeviceInfo.szDeviceName), strlen(pszDeviceName));
        if (0 == strncmp(pszDeviceName, m_tDeviceInfo.szDeviceName, lStingLenMax))
        {
          /* We found it */
          fFound = true;
          break;
        }
        ulCmd = NXDRV_FIND_NEXT;
        printf( "  Command: NXDRV_FIND_NEXT = %d\r\n", ulCmd);
      }
    } while (NXAPI_NO_ERROR == lRet);

    if ( true != fFound)
    {
      printf( "  Device NOT found\r\n");
    }else
    {
      printf( "  Device found\r\n");

      /*----------------------*/
      /* Restart Device       */
      /*----------------------*/
      printf( "\r\n");
      printf( "nxDrvStart()\r\n");
      if( NXAPI_NO_ERROR != (lRet = nxDrvStart ( m_tDeviceInfo.hDevice, 0)))
      {
        ShowError(lRet);
      }else
      {
        printf( "nxDrvStart(): Done\r\n");

        /* Attention, this function does not have any timeouts too verify hardware is corrctly back */
        /* Therefore wait here or brows until device is available again */
        OS_Sleep( 5000);
      }

      /*----------------------------*/
      /* Restart Device with Option */
      /*----------------------------*/
      printf( "\r\n");
      printf( "nxDrvStartEx()\r\n");
      if( NXAPI_NO_ERROR != (lRet = nxDrvStartEx ( m_tDeviceInfo.hDevice, 0, CIFX_TO_FIRMWARE_START, CIFX_RESETEX_SYSTEMSTART)))
      {
        ShowError(lRet);
      }else
      {
        printf( "nxDrvStartEx(): Done\r\n");
      }
    }

    /*----------------------*/
    /* Close nxDrv API      */
    /*----------------------*/
    printf( "\r\n");
    printf( "nxDrvExit()\r\n");
    if( NXAPI_NO_ENTRIES != (lRet = nxDrvExit()))
    {
      ShowError(lRet);
    }else
    {
      printf( "nxDrv API deinitialized\r\n");
    }
  }

  printf( "\r\n");
  printf( "-- nxDrvAPI Test DONE \r\n");

  return fFound;
}


/*****************************************************************************/
/*! Test NXDrv functions to download a firmware
 *  \param  hDevice  Device handle (unused)
 *  \return True on success                                                  */
/*****************************************************************************/
bool nxDrvAPIDownload( char* pszDeviceName)
{
  int32_t lRet   = NXAPI_NO_ERROR;
  bool    fFound = false;

  printf( "\r\n");
  printf( "-- nxDrvAPI Download \r\n");

  printf( "nxDrvInit()\r\n");

  /*----------------------*/
  /* Initialize nxDrv API */
  /*----------------------*/
  if ( NXAPI_NO_ERROR != (lRet = nxDrvInit()))
  {
    ShowError(lRet);
  }else
  {
    uint32_t                  ulCmd       = NXDRV_FIND_FIRST;
    uint32_t                  ulSearchIdx = 0;
    NXDRV_HW_INFORMATION      tDrvInfo;
    NXDRV_DEVICE_INFORMATION  m_tDeviceInfo;

    memset( &tDrvInfo,      0, sizeof( NXDRV_HW_INFORMATION));
    memset( &m_tDeviceInfo, 0, sizeof( NXDRV_DEVICE_INFORMATION));

    printf( "nxDrv API initialized\r\n");

    /* Search the given device in m_tDeviceInfo */
    printf( "\r\n");
    printf( "nxDrvFindDevice(), device with name: %s\r\n", pszDeviceName);
    printf( "  Command: NXDRV_FIND_FIRST = %d\r\n", ulCmd);
    do
    {
      printf( "  Search index = %d\r\n", ulSearchIdx);
      if( NXAPI_NO_ERROR != (lRet = nxDrvFindDevice(ulCmd, sizeof(m_tDeviceInfo), &m_tDeviceInfo, &ulSearchIdx)))
      {
        ShowError(lRet);
      } else
      {
        // Check if we found our device
        long lStingLenMax = max( strlen(m_tDeviceInfo.szDeviceName), strlen(pszDeviceName));
        if (0 == strncmp(pszDeviceName, m_tDeviceInfo.szDeviceName, lStingLenMax))
        {
          /* We found it */
          fFound = true;
          break;
        }
        ulCmd = NXDRV_FIND_NEXT;
        printf( "  Command: NXDRV_FIND_NEXT = %d\r\n", ulCmd);
      }
    } while (NXAPI_NO_ERROR == lRet);

    if ( true != fFound)
    {
      printf( "  Device NOT found\r\n");
    }else
    {
      printf( "  Device found\r\n");

      /*-----------------*/
      /* Load a firmware */
      /*-----------------*/
      char* pszFileName = "D:\\Temp\\X090D001.nxi";
      HANDLE hFile = CreateFile(pszFileName,
                                GENERIC_READ,
                                FILE_SHARE_READ,
                                NULL,
                                OPEN_EXISTING,
                                0,
                                NULL);
      if ( hFile == INVALID_HANDLE_VALUE)
      {
        /* Error opening the file */
        printf("NXDrvAPI: Error opening file <%s>, LastError: %d\r\n", pszFileName, GetLastError());
      } else
      {
        DWORD dwFileSize = GetFileSize(hFile, NULL);
        unsigned char* pabFileData = new unsigned char[dwFileSize];

        DWORD dwBytesRead = 0;
        if ( ReadFile(hFile, pabFileData, dwFileSize, &dwBytesRead, NULL) == false)
        {
          /* Error reading the file */
          printf("NXDrvAPI: Error reading file <%s>, LastError: %d\r\n", pszFileName, GetLastError());
        } else
        {
          lRet = nxDrvDownload( m_tDeviceInfo.hDevice, 0, NXAPI_CMD_FIRMWARE, dwFileSize, pszFileName, pabFileData, NULL, MyProgressCallback);
          if( NXAPI_NO_ERROR != lRet)
          {
            /* Read driver error description */
            ShowError( lRet);
          }else
          {
            /* Start the device */
            lRet = nxDrvStart( m_tDeviceInfo.hDevice, 0);
            if( NXAPI_NO_ERROR != lRet)
            {
              /* Read driver error description */
              ShowError( lRet);
            }
          }
        }

        delete [] pabFileData;
      }
    }

    /*----------------------*/
    /* Close nxDrv API      */
    /*----------------------*/
    printf( "\r\n");
    printf( "nxDrvExit()\r\n");
    if( NXAPI_NO_ENTRIES != (lRet = nxDrvExit()))
    {
      ShowError(lRet);
    }else
    {
      printf( "nxDrv API deinitialized\r\n");
    }
  }

  printf( "\r\n");
  printf( "-- nxDrvAPI Download DONE\r\n");

  return true;
}

/*****************************************************************************/
/*! Entry function for tests on NXAPI                                        */
/*****************************************************************************/
void Test_netXDrvApi( char *pszBoard)
{
  printf("\r\n--- Test the netX driver API ---\r\n");

  /* Test configuration download */
  nxDrvAPITest( pszBoard);

  /* Test configuration download */
  nxDrvAPIDownload( pszBoard);

  printf("\r\n Test the netX driver API done\r\n");
}
